# 画面設計書 6-Task Thread Dump（タスクスレッドダンプ）

## 概要

本ドキュメントは、Apache Spark Web UIの「Task Thread Dump（タスクスレッドダンプ）」画面の設計書である。特定タスクのスレッドダンプ情報（スレッド状態・ロック情報・スタックトレース）を表示する。

### 本画面の処理概要

**業務上の目的・背景**：実行中のタスクがハングアップやデッドロックを起こしている場合に、そのタスクのスレッド状態を詳細に確認するための画面である。スレッドID、スレッド名、スレッド状態、保持しているロック、ブロック元のスレッド情報、完全なスタックトレースを確認できる。パフォーマンスデバッグやデッドロック解析に不可欠な機能を提供する。

**画面へのアクセス方法**：Stage Detail画面のタスク一覧からタスクのスレッドダンプリンクをクリックしてアクセスする。URLパラメータとして`executorId`と`taskId`が必要。`spark.ui.threadDumpsEnabled`設定がtrueの場合のみ有効。

**主要な操作・処理内容**：
1. 指定タスクのスレッドダンプ取得（SparkContext.getTaskThreadDump経由）
2. スレッド情報テーブル表示（Thread ID、Thread Name、Thread State、Thread Locks）
3. ブロック元スレッドへのリンク表示（ブロックされている場合）
4. スタックトレースの表示

**画面遷移**：
- この画面からの遷移先：Executor Thread Dump画面（ブロック元スレッドID選択時）
- この画面への遷移元：Stage Detail画面（タスクのスレッドダンプ表示操作）

**権限による表示制御**：`spark.ui.threadDumpsEnabled`がtrueの場合のみ本画面がStagesTabに登録される。設定がfalseの場合は画面自体が存在しない。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 17 | Spark Web UI | 主機能 | 特定タスクのスレッドダンプ情報（スレッド状態・ロック情報・スタックトレース）を表示する主処理 |
| 5 | Executorプロセス管理 | 主機能 | SparkContext.getTaskThreadDumpを呼び出しExecutor上のタスクスレッド情報を取得 |

## 画面種別

詳細

## URL/ルーティング

- パス: `/stages/taskThreadDump/?executorId={executorId}&taskId={taskId}`
- クラス: `TaskThreadDumpPage` (WebUIPage("taskThreadDump"))
- タブ: `StagesTab` (SparkUITab(parent, "stages"))
- 必須パラメータ: `executorId` (エグゼキュータID、文字列), `taskId` (タスクID、Long)
- 前提条件: `spark.ui.threadDumpsEnabled` = true

## 入出力項目

| 項目名 | 入出力 | 型 | 説明 |
|--------|--------|------|------|
| executorId | 入力（URLパラメータ、必須） | String | スレッドダンプ対象タスクが実行されているエグゼキュータID |
| taskId | 入力（URLパラメータ、必須） | Long | スレッドダンプ対象のタスクID |

## 表示項目

### スレッド情報テーブル

| カラム名 | データソース | 説明 |
|----------|-------------|------|
| Thread ID | thread.threadId | スレッドID |
| Thread Name | thread.threadName | スレッド名 |
| Thread State | thread.threadState | スレッドの状態（RUNNABLE/BLOCKED/WAITING等） |
| Thread Locks | synchronizers + monitors, blockedByThreadId | 保持しているロック情報、ブロック元スレッドへのリンク |

### スタックトレース

`thread.toString` によるスレッドの完全なスタックトレースが `<pre>` タグで表示される。

### 更新日時

`Updated at {formatDate(time)}` で取得時刻が表示される。

## イベント仕様

### 1-ブロック元スレッドリンク

タスクが他のスレッドにブロックされている場合（blockedByThreadId.isDefined）、ブロック元スレッドへのリンクが表示される。リンクはExecutor Thread Dump画面の該当スレッドアンカーに遷移する。遷移先URL: `{uiRoot}/executors/?executorId={executorId}#{blockingThreadId}_td_id`

## データベース更新仕様

### 操作別データベース影響一覧

| 操作（イベント） | 対象テーブル | 操作種別 | 概要 |
|----------------|-------------|---------|------|
| ページ表示 | なし（RPC経由） | - | SparkContext.getTaskThreadDump経由でExecutorからスレッドダンプを取得 |

### テーブル別更新項目詳細

本画面はKVStoreへのアクセスを行わない。SparkContext経由でExecutorにRPCリクエストを送信し、スレッドダンプ情報を取得する。

## メッセージ仕様

| 種別 | メッセージ | 表示条件 |
|------|----------|----------|
| 情報 | "Task {taskId} finished or some error occurred during dumping thread" | スレッドダンプの取得に失敗した場合（タスク終了済みまたはエラー） |
| エラー | "Missing executorId parameter" | executorIdパラメータが未指定の場合 |
| エラー | "Missing taskId parameter" | taskIdパラメータが未指定の場合 |

## 例外処理

| 例外 | 発生条件 | 処理 |
|------|----------|------|
| IllegalArgumentException | executorIdパラメータ未指定 | 例外をスロー |
| IllegalArgumentException | taskIdパラメータ未指定 | 例外をスロー |
| スレッドダンプ取得失敗 | タスク終了済みまたは通信エラー | "Task {taskId} finished or some error occurred"メッセージ表示 |

## 備考

- スレッドダンプの取得はリアルタイムRPC呼び出しであり、KVStoreからのデータ取得ではない
- タスクが既に終了している場合はスレッドダンプを取得できない
- ロック情報はsynchronizers（Lock）とmonitors（Monitor）の2種類を結合して表示する
- ブロック元スレッドリンクのURLにはExecutor Thread Dump画面のアンカーIDが含まれる
- テーブルには`UIUtils.TABLE_CLASS_NOT_STRIPED + " accordion-group"`のCSSクラスが適用される

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ThreadStackTrace | `core/src/main/scala/org/apache/spark/util/ThreadStackTrace.scala` | ThreadStackTraceクラス（threadId, threadName, threadState, blockedByThreadId, blockedByLock, synchronizers, monitors, stackTrace） |

**読解のコツ**: getTaskThreadDumpの戻り値はOption[ThreadStackTrace]。タスクが終了済みの場合Noneが返る。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | TaskThreadDumpPage.scala | `core/src/main/scala/org/apache/spark/ui/jobs/TaskThreadDumpPage.scala` | renderメソッド（39行目）が主処理。109行の小さなクラス |

**主要処理フロー**:
1. **40-50行目**: URLパラメータからexecutorIdとtaskIdを取得（URLデコード付き）
2. **52行目**: 現在時刻を記録
3. **53行目**: `sc.get.getTaskThreadDump(taskId, executorId)` でスレッドダンプを取得
4. **55-106行目**: スレッドダンプ取得成功時: スレッド情報テーブルとスタックトレースを生成。失敗時: エラーメッセージを表示
5. **57-66行目**: blockedByThreadIdが存在する場合、Executor Thread Dumpへのリンクを生成
6. **68-70行目**: synchronizersとmonitorsを結合してロック情報文字列を生成
7. **107行目**: ページタイトル"Thread dump for task {taskId}"でラップ

#### Step 3: スレッドダンプ取得経路を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SparkContext.scala | `core/src/main/scala/org/apache/spark/SparkContext.scala` | getTaskThreadDumpメソッドがExecutorにRPCでスレッドダンプを要求する流れ |

### プログラム呼び出し階層図

```
TaskThreadDumpPage.render(request)
    |
    +-- UIUtils.decodeURLParameter(executorId)
    +-- UIUtils.decodeURLParameter(taskId)
    |
    +-- sc.get.getTaskThreadDump(taskId, executorId)
    |     +-- SparkContext -> ExecutorBackend (RPC)
    |           +-- スレッドダンプ取得
    |
    +-- (成功時) スレッド情報テーブル + スタックトレース生成
    |     +-- executorThreadDumpUrl() ... ブロック元リンク生成
    |
    +-- (失敗時) エラーメッセージ Text 生成
    |
    +-- UIUtils.headerSparkPage()
```

### データフロー図

```
[入力]                       [処理]                          [出力]

HTTPリクエスト
  executorId, taskId  ---> TaskThreadDumpPage.render()
                              |
Executor (RPC)         --> getTaskThreadDump()        --> Option[ThreadStackTrace]
                              |
                              v
                          (Some)  --> テーブル + スタックトレース HTML
                          (None)  --> エラーメッセージ Text
                              |
                              v
                          HTML(Seq[Node])              --> ブラウザ表示
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| TaskThreadDumpPage.scala | `core/src/main/scala/org/apache/spark/ui/jobs/TaskThreadDumpPage.scala` | ソース | 画面のメインページクラス |
| StagesTab.scala | `core/src/main/scala/org/apache/spark/ui/jobs/StagesTab.scala` | ソース | Stagesタブ定義。threadDumpEnabled時のみ本ページを登録 |
| SparkContext.scala | `core/src/main/scala/org/apache/spark/SparkContext.scala` | ソース | getTaskThreadDump APIの定義 |
| ThreadStackTrace.scala | `core/src/main/scala/org/apache/spark/util/ThreadStackTrace.scala` | ソース | スレッドダンプデータモデル |
| UIUtils.scala | `core/src/main/scala/org/apache/spark/ui/UIUtils.scala` | ソース | HTML生成ユーティリティ |
